#include "KumarMedian.hpp"

KumarMedian::KumarMedian(std::function<Metric<Point>*()> createMetric, std::function<Norm<Point>*()> createNorm) :
	metric(createMetric()),
	norm(createNorm()),
	space(createNorm),
	kmedian(createMetric)
{
}

std::unique_ptr<std::vector<Point>> KumarMedian::constructGridSphere(std::vector<Point> const  & basis, Point basePoint, double sideLength, double radius)
{
	std::unique_ptr<std::vector<Point>> gridSphere(new std::vector<Point>());
	long dimension = basis.size();
	std::vector<long> status(dimension);

	// Enumerate all grid points in the first orthant (lying inside the sphere)
	auto increment = [this, &status, &basis, basePoint, dimension, radius, sideLength]() -> bool
	{
		// Determine next point (operates like an adder)
		for(size_t d = 0; d < dimension; ++d)
		{
			++status[d];
			Point candidate(basePoint);
			for(size_t e = 0; e < dimension; ++e)
				candidate += status[e] * sideLength * basis[e];
			double dist = metric->distance(basePoint, candidate);
			if(dist <= radius)
				break;
			else if(d == status.size()-1)
				return false;
			else
				status[d] = 0;
		}
		return true;
	};

	// Add base point to set
	gridSphere->push_back(basePoint);
	
	// Adds all reflection of a grid point in the first orthant
	// with respect to all axies to the set
	while(increment())
	{
		bool pointRemaining = true;
		std::vector<bool> status(dimension);
		while(pointRemaining)
		{
			// Construct reflected point
			Point gridPoint(basePoint);
			for(size_t d = 0; d < dimension; ++d)
				gridPoint += (status[d] ? -1 : 1) * sideLength * basis[d];
			
			// Add point to set
			gridSphere->push_back(gridPoint);
			
			// Determine next reflection (operates like an adder)
			for(size_t d = 0; d < dimension; ++d)
			{
				if(status[d] == false)
				{
					status[d] = true;
					break;
				}
				else if(d == dimension-1)
					pointRemaining = false;
				else
					status[d] = false;
			}
		}
	}

	return gridSphere;
}
